# Portfolio Website - Views Structure Documentation

## 📁 Struktur Folder Views

```
app/Views/
├── layouts/
│   ├── main.php           # Layout utama untuk frontend
│   └── admin.php          # Layout untuk admin panel
├── components/
│   ├── navbar.php         # Navigation bar dengan dropdown
│   └── footer.php         # Footer dengan 4 kolom
├── home/
│   └── index.php          # Halaman beranda
├── auth/
│   └── [login files]      # Halaman autentikasi
├── biodata/
│   └── index.php          # Halaman biodata personal
├── activities/
│   └── index.php          # Halaman aktivitas harian
├── education/
│   └── index.php          # Halaman riwayat pendidikan
├── work/
│   └── index.php          # Halaman pengalaman kerja
├── training/
│   └── index.php          # Halaman pelatihan
├── projects/
│   └── index.php          # Halaman proyek
├── organizations/
│   └── index.php          # Halaman organisasi
├── certificates/
│   └── index.php          # Halaman sertifikat
├── admin/
│   ├── dashboard.php      # Dashboard admin
│   ├── activities/        # CRUD aktivitas
│   ├── biodata/          # CRUD biodata
│   ├── education/        # CRUD pendidikan
│   ├── work/             # CRUD pengalaman kerja
│   ├── training/         # CRUD pelatihan
│   ├── projects/         # CRUD proyek
│   ├── organizations/    # CRUD organisasi
│   └── certificates/     # CRUD sertifikat
└── errors/
    └── [error pages]      # Halaman error
```

---

## 🎨 Design System & Components

### Color Palette
```css
:root {
    --primary-color: #0d6efd;
    --secondary-color: #6c757d;
    --success-color: #198754;
    --danger-color: #dc3545;
    --warning-color: #ffc107;
    --info-color: #0dcaf0;
    --light-color: #f8f9fa;
    --dark-color: #212529;
}
```

### Gradient Themes
- **Primary Gradient**: `linear-gradient(135deg, #667eea 0%, #764ba2 100%)`
- **Hero Background**: `linear-gradient(135deg, #667eea 0%, #764ba2 100%)`
- **Card Shadows**: `0 8px 25px rgba(0,0,0,0.1)`

---

## 📱 Layout Components

### 1. Main Layout (`layouts/main.php`)
```php
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= esc($title ?? 'Portfolio') ?></title>
    <!-- Bootstrap 5.3.0 -->
    <!-- Bootstrap Icons -->
    <!-- Custom CSS -->
</head>
<body>
    <?= $this->include('components/navbar') ?>
    <main>
        <?= $this->renderSection('content') ?>
    </main>
    <?= $this->include('components/footer') ?>
    <!-- Scripts -->
</body>
</html>
```

**Features:**
- Responsive meta viewport
- Bootstrap 5.3.0 integration
- Bootstrap Icons
- Custom CSS/JS integration
- Section-based content rendering

### 2. Navigation Bar (`components/navbar.php`)
```php
<nav class="navbar navbar-expand-lg navbar-dark bg-dark fixed-top shadow-lg">
    <!-- Brand -->
    <a class="navbar-brand fw-bold" href="<?= base_url() ?>">
        <i class="bi bi-person-badge me-2"></i> Portfolio
    </a>
    
    <!-- Mobile Toggle -->
    <button class="navbar-toggler">...</button>
    
    <!-- Navigation Menu -->
    <div class="collapse navbar-collapse">
        <ul class="navbar-nav ms-auto">
            <!-- Home -->
            <li class="nav-item">
                <a class="nav-link" href="<?= base_url() ?>">
                    <i class="bi bi-house-door me-1"></i> Home
                </a>
            </li>
            
            <!-- Biodata -->
            <li class="nav-item">
                <a class="nav-link" href="<?= base_url('biodata') ?>">
                    <i class="bi bi-person me-1"></i> Biodata
                </a>
            </li>
            
            <!-- Activities -->
            <li class="nav-item">
                <a class="nav-link" href="<?= base_url('activities') ?>">
                    <i class="bi bi-calendar-event me-1"></i> Aktivitas
                </a>
            </li>
            
            <!-- Pendidikan Dropdown -->
            <li class="nav-item dropdown">
                <a class="nav-link dropdown-toggle" href="#" data-bs-toggle="dropdown">
                    <i class="bi bi-mortarboard me-1"></i> Pendidikan
                </a>
                <ul class="dropdown-menu dropdown-menu-dark">
                    <li><a class="dropdown-item" href="<?= base_url('education') ?>">
                        <i class="bi bi-mortarboard me-2"></i> Riwayat Pendidikan
                    </a></li>
                    <li><a class="dropdown-item" href="<?= base_url('work') ?>">
                        <i class="bi bi-briefcase me-2"></i> Pengalaman Kerja
                    </a></li>
                    <li><a class="dropdown-item" href="<?= base_url('training') ?>">
                        <i class="bi bi-book me-2"></i> Pelatihan
                    </a></li>
                </ul>
            </li>
            
            <!-- Portfolio Dropdown -->
            <li class="nav-item dropdown">
                <a class="nav-link dropdown-toggle" href="#" data-bs-toggle="dropdown">
                    <i class="bi bi-collection me-1"></i> Portfolio
                </a>
                <ul class="dropdown-menu dropdown-menu-dark">
                    <li><a class="dropdown-item" href="<?= base_url('projects') ?>">
                        <i class="bi bi-folder me-2"></i> Proyek
                    </a></li>
                    <li><a class="dropdown-item" href="<?= base_url('certificates') ?>">
                        <i class="bi bi-award me-2"></i> Sertifikat
                    </a></li>
                    <li><a class="dropdown-item" href="<?= base_url('organizations') ?>">
                        <i class="bi bi-people me-2"></i> Organisasi
                    </a></li>
                </ul>
            </li>
            
            <!-- Admin Login -->
            <li class="nav-item ms-2">
                <a class="btn btn-primary btn-sm text-white" href="<?= base_url('auth/login') ?>">
                    <i class="bi bi-box-arrow-in-right me-1"></i> Admin
                </a>
            </li>
        </ul>
    </div>
</nav>
```

**Features:**
- Fixed top navigation
- Responsive collapse menu
- Dropdown menus for grouped items
- Active state highlighting
- Bootstrap Icons integration
- Hover effects and animations

### 3. Footer (`components/footer.php`)
```php
<footer class="bg-gradient-dark text-light py-5 mt-5">
    <div class="container">
        <div class="row">
            <!-- Brand Section -->
            <div class="col-lg-4 col-md-6 mb-4 mb-lg-0">
                <h4 class="fw-bold mb-3">
                    <div class="bg-primary rounded-circle p-2 me-3">
                        <i class="bi bi-person-badge text-white"></i>
                    </div>
                    Portfolio
                </h4>
                <p class="text-light opacity-75 mb-4">...</p>
                
                <!-- Social Media -->
                <div class="social-links">
                    <h6 class="fw-semibold mb-3">Ikuti Saya</h6>
                    <div class="d-flex gap-3">
                        <a href="#" class="social-link">
                            <i class="bi bi-linkedin"></i>
                        </a>
                        <!-- More social links -->
                    </div>
                </div>
            </div>
            
            <!-- Quick Links -->
            <div class="col-lg-3 col-md-6 mb-4 mb-lg-0">
                <h5 class="fw-bold mb-4">Quick Links</h5>
                <ul class="list-unstyled footer-links">
                    <!-- Navigation links -->
                </ul>
            </div>
            
            <!-- Features -->
            <div class="col-lg-2 col-md-6 mb-4 mb-lg-0">
                <h5 class="fw-bold mb-4">Fitur</h5>
                <ul class="list-unstyled footer-links">
                    <!-- Feature links -->
                </ul>
            </div>
            
            <!-- Contact Info -->
            <div class="col-lg-3 col-md-6">
                <h5 class="fw-bold mb-4">Kontak</h5>
                <div class="contact-info">
                    <!-- Contact details -->
                </div>
            </div>
        </div>
        
        <!-- Footer Bottom -->
        <div class="footer-bottom mt-5 pt-4">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <p class="mb-0 text-light opacity-75">
                        &copy; <?= date('Y') ?> Portfolio. Dibuat dengan ❤️ menggunakan CodeIgniter 4
                    </p>
                </div>
                <div class="col-md-6 text-md-end mt-3 mt-md-0">
                    <div class="footer-badges">
                        <span class="badge bg-primary me-2">CodeIgniter 4</span>
                        <span class="badge bg-success me-2">Bootstrap 5</span>
                        <span class="badge bg-info">Responsive</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</footer>
```

**Features:**
- 4-column layout (Brand, Quick Links, Features, Contact)
- Gradient background
- Social media links with hover effects
- Technology badges
- Responsive design

---

## 🏠 Frontend Pages

### 1. Home Page (`home/index.php`)
```php
<?= $this->extend('layouts/main') ?>
<?= $this->section('content') ?>

<!-- Hero Section -->
<div class="hero-section">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-lg-6 text-start">
                <!-- Profile info from biodata -->
                <h1 class="display-4 fw-bold mb-4"><?= esc($biodata['nama_lengkap']) ?></h1>
                <p class="lead mb-4"><?= esc($biodata['nama_panggilan']) ?></p>
                <p class="mb-4"><?= esc($biodata['tentang_saya']) ?></p>
                
                <!-- Action buttons -->
                <div class="d-flex gap-3">
                    <a href="mailto:<?= esc($biodata['email']) ?>" class="btn btn-light btn-lg">
                        <i class="bi bi-envelope me-2"></i>Email
                    </a>
                    <a href="<?= esc($biodata['linkedin']) ?>" class="btn btn-outline-light btn-lg">
                        <i class="bi bi-linkedin me-2"></i>LinkedIn
                    </a>
                </div>
            </div>
            <div class="col-lg-6 text-center">
                <!-- Profile image -->
                <img src="<?= base_url('writable/uploads/biodata/' . $biodata['foto_profil']) ?>" 
                     class="img-fluid rounded-circle shadow-lg" 
                     style="max-width: 300px; border: 5px solid white;">
            </div>
        </div>
    </div>
</div>

<!-- Recent Activities Section -->
<div class="container py-5">
    <h2 class="section-title">
        <i class="bi bi-calendar-event me-2"></i>Aktivitas Terbaru
    </h2>
    <div class="row">
        <?php foreach ($recentActivities as $activity): ?>
            <div class="col-md-4 mb-4">
                <div class="card h-100">
                    <!-- Activity card content -->
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Recent Certificates Section -->
<div class="container py-5">
    <h2 class="section-title">
        <i class="bi bi-award me-2"></i>Sertifikat Terbaru
    </h2>
    <div class="row">
        <?php foreach ($recentCertificates as $cert): ?>
            <div class="col-md-4 mb-4">
                <div class="card h-100 certificate-card-home">
                    <!-- Certificate card with hover effects -->
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<?= $this->endSection() ?>
```

**Features:**
- Hero section with profile info and image
- Recent activities grid (3 items)
- Recent certificates grid (3 items)
- Responsive card layouts
- Call-to-action buttons

### 2. Certificates Page (`certificates/index.php`)
```php
<?= $this->extend('layouts/main') ?>
<?= $this->section('content') ?>

<div class="container py-5">
    <h1 class="section-title mb-4">
        <i class="bi bi-award me-2"></i>Sertifikat
    </h1>

    <!-- Enhanced Search and Filter Section -->
    <div class="search-filter-section">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h4 class="mb-0">
                <i class="bi bi-funnel me-2 text-primary"></i>Filter & Pencarian
            </h4>
            <div class="text-muted">
                <i class="bi bi-collection me-1"></i>
                Total: <strong><?= $total ?></strong> sertifikat
            </div>
        </div>
        
        <form method="GET" class="row g-3">
            <!-- Search input -->
            <div class="col-md-4">
                <label class="form-label">
                    <i class="bi bi-search me-2"></i>Cari Sertifikat
                </label>
                <div class="input-group">
                    <input type="text" class="form-control" name="search" 
                           placeholder="Nama sertifikat, penerbit, atau deskripsi...">
                    <button class="btn btn-outline-secondary" type="button">
                        <i class="bi bi-x"></i>
                    </button>
                </div>
            </div>
            
            <!-- Filter inputs -->
            <div class="col-md-2">
                <label class="form-label">Penerbit</label>
                <input type="text" class="form-control" name="penerbit">
            </div>
            
            <!-- Date filters -->
            <div class="col-md-2">
                <label class="form-label">Dari Tanggal</label>
                <input type="date" class="form-control" name="tanggal_dari">
            </div>
            
            <!-- Sort options -->
            <div class="col-md-1">
                <label class="form-label">Urutkan</label>
                <select class="form-select" name="sort">
                    <option value="tanggal_terbit">Tanggal</option>
                    <option value="nama_sertifikat">Nama</option>
                    <option value="penerbit">Penerbit</option>
                </select>
            </div>
            
            <!-- Action buttons -->
            <div class="col-12 d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-search me-1"></i>Cari & Filter
                </button>
                <a href="<?= base_url('certificates') ?>" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-clockwise me-1"></i>Reset
                </a>
            </div>
        </form>
    </div>

    <!-- Certificates Grid -->
    <div class="certificate-grid">
        <?php foreach ($certificates as $cert): ?>
            <div class="certificate-card">
                <!-- Certificate image -->
                <img src="<?= base_url('writable/uploads/certificates/' . $cert['foto_sertifikat']) ?>" 
                     class="certificate-img">
                
                <div class="p-3">
                    <h5 class="mb-2"><?= esc($cert['nama_sertifikat']) ?></h5>
                    
                    <!-- Certificate details -->
                    <p class="text-muted mb-2">
                        <i class="bi bi-building me-1"></i><?= esc($cert['penerbit']) ?>
                    </p>
                    <p class="text-muted mb-2">
                        <i class="bi bi-calendar me-1"></i>
                        <?= date('d F Y', strtotime($cert['tanggal_terbit'])) ?>
                    </p>
                    
                    <!-- Action button -->
                    <a href="<?= esc($cert['link_sertifikat']) ?>" target="_blank" 
                       class="btn btn-sm btn-primary w-100 mt-2">
                        <i class="bi bi-link-45deg me-1"></i>Lihat Sertifikat
                    </a>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <!-- Enhanced Pagination -->
    <nav aria-label="Page navigation" class="mt-5">
        <!-- Pagination info -->
        <div class="d-flex justify-content-between align-items-center mb-3">
            <div class="pagination-info">
                <small class="text-muted">
                    Menampilkan <?= (($currentPage - 1) * 5) + 1 ?> - <?= min($currentPage * 5, $total) ?> dari <?= $total ?> sertifikat
                </small>
            </div>
            <div class="pagination-summary">
                <small class="text-muted">
                    Halaman <?= $currentPage ?> dari <?= $lastPage ?>
                </small>
            </div>
        </div>
        
        <!-- Pagination controls -->
        <ul class="pagination justify-content-center">
            <!-- Previous/Next buttons with page numbers -->
        </ul>
        
        <!-- Quick jump -->
        <div class="text-center mt-3">
            <form method="GET" class="d-inline-flex align-items-center">
                <small class="text-muted me-2">Lompat ke halaman:</small>
                <input type="number" name="page" min="1" max="<?= $lastPage ?>" 
                       class="form-control form-control-sm" style="width: 80px;">
            </form>
        </div>
    </nav>
</div>

<?= $this->endSection() ?>
```

**Features:**
- Advanced search and filter system
- Certificate grid layout (5 items per page)
- Enhanced pagination with info and quick jump
- Responsive design
- Image hover effects

### 3. Education Page (`education/index.php`)
```php
<?= $this->extend('layouts/main') ?>
<?= $this->section('content') ?>

<div class="container py-5">
    <h1 class="section-title mb-4">
        <i class="bi bi-mortarboard me-2"></i>Riwayat Pendidikan
    </h1>

    <!-- Search and Filter Section -->
    <div class="search-filter-section">
        <form method="GET" class="row g-3">
            <!-- Search and filter inputs -->
        </form>
    </div>

    <!-- Education Timeline -->
    <div class="education-timeline">
        <?php foreach ($education as $edu): ?>
            <div class="education-item">
                <div class="d-flex justify-content-between align-items-start mb-2">
                    <div>
                        <h4 class="mb-1">
                            <i class="bi bi-mortarboard-fill me-2 text-primary"></i>
                            <?= esc($edu['nama_sekolah']) ?>
                        </h4>
                        
                        <!-- Badges -->
                        <span class="badge bg-primary"><?= esc($edu['jenjang']) ?></span>
                        <?php if ($edu['jurusan']): ?>
                            <span class="badge bg-secondary ms-2"><?= esc($edu['jurusan']) ?></span>
                        <?php endif; ?>
                    </div>
                    
                    <div class="text-end">
                        <!-- Status Badge -->
                        <?php 
                        $status = $edu['status_pendidikan'] ?? 'lulus';
                        if (empty($edu['tahun_selesai']) && $status !== 'sedang_menempuh') {
                            $status = 'sedang_menempuh';
                        }
                        ?>
                        
                        <?php if ($status === 'sedang_menempuh'): ?>
                            <span class="badge bg-warning text-dark mb-2">
                                <i class="bi bi-clock me-1"></i>Sedang Menempuh
                            </span>
                        <?php elseif ($status === 'lulus'): ?>
                            <span class="badge bg-success mb-2">
                                <i class="bi bi-check-circle me-1"></i>Lulus
                            </span>
                        <?php elseif ($status === 'tidak_lulus'): ?>
                            <span class="badge bg-danger mb-2">
                                <i class="bi bi-x-circle me-1"></i>Tidak Lulus
                            </span>
                        <?php endif; ?>
                        
                        <!-- Period -->
                        <?php if ($edu['tahun_mulai'] && $edu['tahun_selesai']): ?>
                            <p class="mb-0 text-muted">
                                <i class="bi bi-calendar-range me-1"></i>
                                <?= esc($edu['tahun_mulai']) ?> - <?= esc($edu['tahun_selesai']) ?>
                            </p>
                        <?php elseif ($edu['tahun_mulai']): ?>
                            <p class="mb-0 text-muted">
                                <i class="bi bi-calendar me-1"></i>
                                <?= $status === 'sedang_menempuh' ? 'Mulai: ' . esc($edu['tahun_mulai']) . ' (Berlangsung)' : 'Mulai: ' . esc($edu['tahun_mulai']) ?>
                            </p>
                        <?php endif; ?>
                        
                        <!-- Grade -->
                        <?php if ($edu['nilai_akhir']): ?>
                            <p class="mb-0">
                                <strong>
                                    <i class="bi bi-trophy me-1 text-warning"></i>
                                    Nilai: <?= esc($edu['nilai_akhir']) ?>
                                </strong>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Description -->
                <?php if ($edu['deskripsi']): ?>
                    <p class="text-muted mb-2"><?= esc($edu['deskripsi']) ?></p>
                <?php endif; ?>
                
                <!-- Certificate link -->
                <?php if ($edu['sertifikat']): ?>
                    <a href="<?= base_url('writable/uploads/education/' . $edu['sertifikat']) ?>" 
                       target="_blank" class="btn btn-sm btn-outline-primary">
                        <i class="bi bi-file-earmark-pdf me-1"></i>Lihat Sertifikat
                    </a>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<?= $this->endSection() ?>
```

**Features:**
- Timeline layout with visual indicators
- Status badges (Lulus, Sedang Menempuh, Tidak Lulus)
- Responsive design
- Search and filter functionality
- Certificate download links

---

## 🔧 Admin Panel

### Admin Layout (`layouts/admin.php`)
```php
<!DOCTYPE html>
<html lang="id">
<head>
    <!-- Similar to main layout but with admin-specific styles -->
</head>
<body>
    <?= $this->include('components/admin_navbar') ?>
    
    <div class="admin-wrapper">
        <main class="admin-content">
            <?= $this->renderSection('content') ?>
        </main>
    </div>
    
    <!-- Admin-specific scripts -->
</body>
</html>
```

### Admin Education Create (`admin/education/create.php`)
```php
<?= $this->extend('layouts/admin') ?>
<?= $this->section('content') ?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2 class="mb-0">
        <i class="bi bi-plus-circle me-2"></i>Tambah Riwayat Pendidikan
    </h2>
    <a href="<?= base_url('admin/education') ?>" class="btn btn-secondary">
        <i class="bi bi-arrow-left me-1"></i>Kembali
    </a>
</div>

<div class="admin-card">
    <form action="<?= base_url('admin/education/store') ?>" method="POST" enctype="multipart/form-data">
        <?= csrf_field() ?>
        
        <div class="row">
            <!-- Jenjang -->
            <div class="col-md-6 mb-3">
                <label for="jenjang" class="form-label">Jenjang <span class="text-danger">*</span></label>
                <select class="form-select" id="jenjang" name="jenjang" required>
                    <option value="">Pilih Jenjang</option>
                    <option value="SD">SD</option>
                    <option value="SMP">SMP</option>
                    <option value="SMA">SMA</option>
                    <option value="SMK">SMK</option>
                    <option value="D3">D3 (Diploma)</option>
                    <option value="S1">S1 (Sarjana)</option>
                    <option value="S2">S2 (Magister)</option>
                    <option value="S3">S3 (Doktor)</option>
                </select>
            </div>
            
            <!-- Urutan -->
            <div class="col-md-6 mb-3">
                <label for="urutan" class="form-label">
                    <i class="bi bi-sort-numeric-up me-1"></i>Urutan Tampil
                </label>
                <input type="number" class="form-control" id="urutan" name="urutan" 
                       value="0" min="0" max="100">
                <div class="form-text">
                    <i class="bi bi-info-circle me-1"></i>
                    <strong>Urutan untuk menampilkan data:</strong><br>
                    • 0 = Tampil paling atas/pertama<br>
                    • 1, 2, 3... = Urutan selanjutnya<br>
                    • Semakin kecil angka, semakin atas posisinya
                </div>
            </div>
        </div>

        <!-- Nama Sekolah -->
        <div class="mb-3">
            <label for="nama_sekolah" class="form-label">Nama Sekolah/Universitas <span class="text-danger">*</span></label>
            <input type="text" class="form-control" id="nama_sekolah" name="nama_sekolah" required>
        </div>

        <!-- Jurusan -->
        <div class="mb-3">
            <label for="jurusan" class="form-label">Jurusan</label>
            <input type="text" class="form-control" id="jurusan" name="jurusan" 
                   placeholder="Contoh: Teknik Informatika">
        </div>

        <div class="row">
            <!-- Tahun Mulai -->
            <div class="col-md-3 mb-3">
                <label for="tahun_mulai" class="form-label">Tahun Mulai</label>
                <input type="number" class="form-control" id="tahun_mulai" name="tahun_mulai" 
                       min="1900" max="2100" placeholder="Contoh: 2015">
            </div>
            
            <!-- Status -->
            <div class="col-md-3 mb-3">
                <label for="status_pendidikan" class="form-label">Status</label>
                <select class="form-select" id="status_pendidikan" name="status_pendidikan" onchange="toggleTahunSelesai()">
                    <option value="lulus">Lulus</option>
                    <option value="sedang_menempuh">Sedang Menempuh</option>
                    <option value="tidak_lulus">Tidak Lulus</option>
                </select>
            </div>
            
            <!-- Tahun Selesai -->
            <div class="col-md-3 mb-3" id="tahun_selesai_container">
                <label for="tahun_selesai" class="form-label">Tahun Selesai</label>
                <input type="number" class="form-control" id="tahun_selesai" name="tahun_selesai" 
                       min="1900" max="2100" placeholder="Contoh: 2019">
                <small class="text-muted">Kosongkan jika sedang menempuh</small>
            </div>
            
            <!-- Nilai Akhir -->
            <div class="col-md-3 mb-3">
                <label for="nilai_akhir" class="form-label">Nilai Akhir</label>
                <input type="number" step="0.01" class="form-control" id="nilai_akhir" name="nilai_akhir" 
                       placeholder="Contoh: 3.75 atau 90.5">
                <small class="text-muted">IPK/Rata-rata nilai</small>
            </div>
        </div>

        <!-- Deskripsi -->
        <div class="mb-3">
            <label for="deskripsi" class="form-label">Deskripsi</label>
            <textarea class="form-control" id="deskripsi" name="deskripsi" rows="5"></textarea>
        </div>

        <!-- Sertifikat -->
        <div class="mb-3">
            <label for="sertifikat" class="form-label">Sertifikat</label>
            <input type="file" class="form-control" id="sertifikat" name="sertifikat" 
                   accept="image/jpeg,image/png,application/pdf">
            <small class="text-muted">Format: JPG, PNG, PDF (Max: 5MB)</small>
        </div>

        <!-- Action Buttons -->
        <div class="d-flex gap-2">
            <button type="submit" class="btn btn-primary">
                <i class="bi bi-save me-1"></i>Simpan
            </button>
            <a href="<?= base_url('admin/education') ?>" class="btn btn-secondary">
                <i class="bi bi-x me-1"></i>Batal
            </a>
        </div>
    </form>
</div>

<script>
function toggleTahunSelesai() {
    const status = document.getElementById('status_pendidikan').value;
    const tahunSelesaiContainer = document.getElementById('tahun_selesai_container');
    const tahunSelesaiInput = document.getElementById('tahun_selesai');
    
    if (status === 'sedang_menempuh') {
        tahunSelesaiContainer.style.opacity = '0.5';
        tahunSelesaiInput.disabled = true;
        tahunSelesaiInput.value = '';
        tahunSelesaiInput.placeholder = 'Tidak diperlukan (sedang menempuh)';
    } else {
        tahunSelesaiContainer.style.opacity = '1';
        tahunSelesaiInput.disabled = false;
        tahunSelesaiInput.placeholder = 'Contoh: 2019';
    }
}

document.addEventListener('DOMContentLoaded', function() {
    toggleTahunSelesai();
});
</script>

<?= $this->endSection() ?>
```

**Features:**
- Comprehensive form with validation
- Status-based field toggling
- Clear field explanations
- File upload support
- Responsive layout

---

## 📱 Responsive Design Features

### Mobile Navigation
- Collapsible hamburger menu
- Touch-friendly dropdown menus
- Swipe gestures support

### Grid Systems
- Bootstrap 5 responsive grid
- Flexible card layouts
- Mobile-first approach

### Typography
- Scalable font sizes
- Readable line heights
- Proper contrast ratios

### Interactive Elements
- Touch-friendly buttons (min 44px)
- Hover states for desktop
- Loading states for forms
- Smooth animations

---

## 🎯 Key UI/UX Patterns

### Card Components
```css
.card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 25px rgba(0,0,0,0.15);
}
```

### Button Styles
```css
.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
    border-radius: 25px;
    padding: 10px 30px;
    transition: all 0.3s ease;
}
```

### Form Elements
```css
.form-control {
    border-radius: 10px;
    border: 2px solid #e9ecef;
    transition: all 0.3s ease;
}

.form-control:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}
```

---

## 📊 Performance Considerations

### Image Optimization
- Lazy loading for certificate images
- Responsive image sizes
- WebP format support

### CSS/JS Optimization
- Minified assets
- Critical CSS inlining
- Async script loading

### Accessibility
- ARIA labels
- Keyboard navigation
- Screen reader support
- Color contrast compliance

---

## 🔄 Interactive Features

### Pagination
- 5 items per page for certificates
- Quick jump functionality
- Keyboard navigation (Ctrl+Arrow)

### Search & Filter
- Real-time search suggestions
- Advanced filtering options
- URL-based state management

### Animations
- Smooth page transitions
- Hover effects
- Loading states
- Scroll animations

---

Dokumentasi ini memberikan gambaran lengkap tentang struktur views dan komponen UI yang ada dalam aplikasi portfolio. Semua komponen dirancang dengan prinsip mobile-first, accessibility, dan user experience yang optimal.